#!/usr/bin/env bash

. PlotCommon.sh NoPreamble

Ensembles="${Ensembles:-$(echo C{1,2} M{1,2,3} F1M)}"
Series=${Suffix:-renorm}
Heavy=${Heavy:-D_s}

# Assumes $PWD is NoSync
NoSync=$PWD
MikeThesis=$(dirname $(readlink -f R $0)); MikeThesis=${MikeThesis%/Scripts}
PlotLinksSub='ThesisLinks/ZV'
PlotLinksPrefix='../../'
mkdir -p "$PlotLinksSub"
PlotLinks="$PlotLinksSub/${Series}_"
ZVSummary="${PlotLinks}ZVSummary.tex"

function MakeLink()
{
local Dest="${PlotLinks}$1"
local Source="${PlotLinksPrefix}$2"
local Extra="$3"
local Suffix="${Source##*.model}"
local Prefix="${Source%$Suffix}"
      Suffix="${Suffix%h5}pdf"
if [ -e "$Dest" ] || [ -h "$Dest" ]; then rm "$Dest"; fi
ln -s "$Prefix$Extra$Suffix" "$Dest"
}

function ParseEnsembleZV()
{
  # Get ZV_mixed before I cd
  local -a aZV aE aFileParts
  local Quark File Comment Key Prefix Action Tmp FileParts
  local InDir="$Ensemble/Renorm"
  aZV=($(GetColumn --exact ZV $Ensemble/FormFactor/${Series}_mostly/3sm_sp2/F3_l_h*_p2_0.g5P_g5W.model.*.h5))
  ZV[${Ensemble}_'m']="${aZV[1]}"

  while read -r Quark File Comment || [ -n "$Quark" ]
  do
    FileParts="${File##*/}"
    FileParts="${FileParts%%.*}"
    aFileParts=(${FileParts//_/ })
    Action="${Quark:0:1}"
    if [ "$Action" != h ] && [ "$Action" != l ]; then
      echo "  Unrecognised quark action \"$Quark\""
    elif (( "${#aFileParts[@]}" < 4 || "${aFileParts[2]}" != dt )); then
      echo "  File dt missing \"$File\""
    else
    File="$InDir/$File"
    Key=${Ensemble}_${Action}
    DeltaT[$Key]="${aFileParts[3]}"
    # Save ZV
    aZV=($(GetColumn --exact ZV "$File"))
    ZV[$Key]="${aZV[1]}"
    echo "$Quark $Action ${ZV[$Key]} $Comment"
    echo "  Fit $File"
    # Get ratio file this ZV came from
    eval RatioFile="$InDir/"$(h5dump -d /model/FileList "$File" | grep : | cut -w -f 3-)
    echo "  Ratio   $RatioFile"
    eval RSource=($(h5dump -d /fold/FileList "$RatioFile" | grep : | cut -w -f 3-))
    # Get energy which entered the ratio
    Energy="$InDir/${RSource[2]}"
    echo "  E_0 $Energy"
    aE=($(GetColumn --exact pvalueH,E "$Energy"))
    pval[$Key]="${aE[1]%(*}"
    E[$Key]="${aE[9]}"
    # Save spectator
    Tmp=${RatioFile%/*}
    Tmp=${Tmp##*/}
    case "${Tmp:0:1}" in
      h) Spec[$Key]="heavy";;
      l) Spec[$Key]="light";;
      s) Spec[$Key]="strange";;
      *) Spec[$Key]="${Tmp:0:1}";;
    esac
    # Save Decay
    Tmp=${RatioFile##*/}
    Tmp=${Tmp#*_}
    case "${Tmp:0:1}" in
      h) Decay[$Key]="heavy";;
      l) Decay[$Key]="light";;
      s) Decay[$Key]="strange";;
      *) Decay[$Key]="${Tmp:0:1}";;
    esac
    # Replot
    PlotZVFit "$File" > /dev/null
    # Make links
    MakeLink "${Key}_Energy.pdf" "$Energy" _log
    MakeLink "${Key}_Fit.pdf" "$File" _corr
    fi
  done < "$InDir/ZV$Suffix.txt"
}

function WriteZVTable()
{
  local Q EnsCode Prefix ActL ActH ActM
  # Write declarations
  for Ensemble in $Ensembles
  do
    ConvertDigits "$Ensemble"
    Prefix='\def\ValZV'"${Series}${EnsCode}"
    for Q in l h
    do
      ActL="${Ensemble}_${Q}"
      echo "${Prefix}${Q}ZV{${ZV[$ActL]}}"
      echo "${Prefix}${Q}E{${E[$ActL]}}"
      echo "${Prefix}${Q}PVal{${pval[$ActL]}}"
      echo "${Prefix}${Q}Decay{${Decay[$ActL]}}"
      echo "${Prefix}${Q}Spec{${Spec[$ActL]}}"
      echo "${Prefix}${Q}DeltaT{${DeltaT[$ActL]}}"
    done
    echo "${Prefix}mZV{${ZV[${Ensemble}_m]}}"
  done
  # Write header
  echo '\begin{table}[H]'
  echo '\smaller'
  echo '\centering'
  echo '\begin{tabular}{c|ll|l|ll|l|l}'
  echo '\toprule'
  echo '& \multicolumn{3}{c|}{heavy} & \multicolumn{3}{c|}{light} & \multicolumn{1}{c}{mixed} \\'
  echo 'Name & $a E_0$ $['"$Heavy"']$ & $p$-value & $Z_V$ & $a E_0$ $[K]$ & $p$-value & $Z_V$ & $Z_V$ \\'
  #echo 'Name & \multicolumn{1}{c}{$a E_0$} & \multicolumn{1}{c|}{$p$-val} & \multicolumn{1}{c|}{$Z_V$} & \multicolumn{1}{c}{$a E_0$} & \multicolumn{1}{c|}{$p$-val} & \multicolumn{1}{c|}{$Z_V$} & \multicolumn{1}{c}{$Z_V$} \\'
  echo '\midrule'
  # Now write data for each ensemble
  for Ensemble in $Ensembles
  do
    ActL="${Ensemble}_l"
    ActH="${Ensemble}_h"
    ActM="${Ensemble}_m"
    echo "$Ensemble & ${E[$ActH]} & ${pval[$ActH]} & ${ZV[$ActH]} & ${E[$ActL]} & ${pval[$ActL]} & ${ZV[$ActL]} & ${ZV[$ActM]}"' \\'
  done
  # Write footer
  echo '\bottomrule'
  echo '\end{tabular}'
}

############################################################

# Main loop

############################################################

declare -A ZV E pval Decay Spec DeltaT

# Write each row
for Ensemble in $Ensembles
do
  echo -e "----------\n$Ensemble"
  ParseEnsembleZV
done

WriteZVTable > "$ZVSummary"

#echo -e "----------\nMake sure you've done this at least once:"
#echo 'for E in '"$Ensembles"'; do DoEFit= FitZV$E.sh; done'
