#!/usr/bin/env bash

. PlotCommon.sh NoPreamble

Ensembles="${Ensembles:-$(echo C{1,2} M{1,2,3} F1M)}"

# Assumes $PWD is NoSync
NoSync=$PWD
MikeThesis=$(dirname $(readlink -f R $0)); MikeThesis=${MikeThesis%/Scripts}
PlotLinks='ThesisLinks/mPi/'
PlotLinksPrefix='../../'
mkdir -p "$PlotLinks"
Summary="$PlotLinks/mPiSummary.tex"

function MakeLink()
{
local Dest="${PlotLinks}$1"
local Source="${PlotLinksPrefix}$2"
local Extra="$3"
local Suffix="${Source##*.model}"
local Prefix="${Source%$Suffix}"
      Suffix="${Suffix%h5}pdf"
if [ -e "$Dest" ] || [ -h "$Dest" ]; then rm "$Dest"; fi
ln -s "$Prefix$Extra$Suffix" "$Dest"
}

function ParseMPi()
{
  local -a aValue
  local Ensemble File pH nE E0 E1 i
  while read -r Ensemble File || [ -n "$Ensemble" ]
  do
    aValue=($(GetColumn --exact pvalueH,E "$File"))
    pH="${aValue[0*8+1]%(*}"
    E0="${aValue[1*8+1]}"
    nE=$(( ${#aValue[@]} / 8 - 1 ))
    if (( nE<2 )); then unset E1; else E1="${aValue[2*8+1]}"; fi
    echo "$Ensemble $nE $E0 $E1 $pH ${File##*/}"
    Value[${Ensemble}'_pH']="$pH"
    Value[${Ensemble}'_aE0']="$E0"
    Value[${Ensemble}'_aE1']="$E1"
    Value[${Ensemble}'_nE']="$nE"
    # Make links
    MakeLink "${Ensemble}_mPi.pdf" "${File##*NoSync/}" _log
  done < "$MLUCache/mPi.txt"
  # Now get the masses in physical units
  aValue=($(GetColumn --partial=-mP "$MLUCache/EnsembleInfo.h5"))
  for (( i=1;i<${#aValue[@]};i+=8 ))
  do
    Ensemble="${aValue[i-1]%%-*}"
    E0="${aValue[i]}"
    if [ "${E0: -2}" == E8 ]
    then
      E1="${E0:0:-2}" # Get rid of exp
      E0="${E1%%(*}" # Numeric part only
      pH=${#E0}
      nE="${E1:pH}" # error
      E0="$(bc <<<"scale=$((pH-4));${E0}*100/1")${nE:0:2}.${nE:2}"
    fi
    Value[${Ensemble}'_m']="$E0"
  done
  # Now add reference values from https://arxiv.org/pdf/1812.08791.pdf \cite{Boyle:2018knm}, Table 1, pg 6
  Value['C1_mRef']='339.76(1.22)'
  Value['C2_mRef']='430.63(1.38)'
  Value['M1_mRef']='303.56(1.38)'
  Value['M2_mRef']='360.71(1.58)'
  Value['M3_mRef']='410.76(1.74)'
  Value['F1M_mRef']='232.01(1.01)'
}

function WriteMPi()
{
  local Q EnsCode Prefix
  # Write declarations
  for Ensemble in $Ensembles
  do
    ConvertDigits "$Ensemble"
    Prefix='\def\ValmPi'"${EnsCode}"
    echo "${Prefix}${Q}pH{${Value[${Ensemble}'_pH']}}"
    echo "${Prefix}${Q}nE{${Value[${Ensemble}'_nE']}}"
    echo "${Prefix}${Q}aEZero{${Value[${Ensemble}'_aE0']}}"
    echo "${Prefix}${Q}aEOne{${Value[${Ensemble}'_aE1']}}"
    echo "${Prefix}${Q}m{${Value[${Ensemble}'_m']}}"
    echo "${Prefix}${Q}mRef{${Value[${Ensemble}'_mRef']}}"
  done
  # Write header
  echo '\begin{table}[H]'
  echo '\centering'
  echo '\begin{tabular}{clll|ll}'
  echo '\toprule'
  echo '&&&& \multicolumn{2}{c}{$m_\pi$/MeV} \\'
  echo 'Name & $n_{\textrm{state}}$ & $a m_\pi$ & $p$-value & fit & ref \cite{Boyle:2018knm} \\'
  echo '\midrule'
  # Now write data for each ensemble
  for Ensemble in $Ensembles
  do
    echo "$Ensemble & ${Value[${Ensemble}'_nE']} & ${Value[${Ensemble}'_aE0']} & ${Value[${Ensemble}'_pH']} & ${Value[${Ensemble}'_m']} & ${Value[${Ensemble}'_mRef']}"' \\'
  done
  # Write footer
  echo '\bottomrule'
  echo '\end{tabular}'
}

############################################################

# Main loop

############################################################

declare -A Value

# Write each row
ParseMPi
WriteMPi > "$Summary"

echo -e "----------\nMake sure you've done this at least once:"
echo 'for E in '"$Ensembles"'; do PlotOnly= DoEFit= FitZV$E.sh; done'
